#define TIME_FOR_PAIR_OF_KEY_AND_LOCK 0
const bool NEED_ANSWER_RESTORATION = false;

#include <iostream>
#include <vector>
#include <algorithm>

using Coordinate = int32_t;
using PathLength = int64_t;
using Index = uint32_t;

const PathLength IMPOSSIBLE_LENGTH = -1;

struct KeyLock {
    Coordinate key, lock;
};

std::istream& operator>>(std::istream& in, KeyLock& key_lock) {
    return in >> key_lock.key >> key_lock.lock;
}

struct Instance {
    std::vector<KeyLock> key_locks;
    Coordinate start, finish;
};

std::istream& operator>>(std::istream& in, Instance& instance) {
    Index n = 0;
    in >> n >> instance.start >> instance.finish;
    instance.key_locks.resize(n);
    for (KeyLock& key_lock : instance.key_locks)
        in >> key_lock;
    return in;
}

enum ActionType {
    GO, PICK, UNLOCK
};

std::ostream& operator<<(std::ostream& out, const ActionType action_type) {
    if (action_type == GO)
        out << "go";
    else if (action_type == PICK)
        out << "pick";
    else
        out << "unlock";
    return out;
}

struct Action {
    ActionType type;
    Coordinate x;
};

std::ostream& operator<<(std::ostream& out, const Action action) {
    return out << action.type << ' ' << action.x;
}

struct Actions {
    std::vector<Action> v;
};

std::ostream& operator<<(std::ostream& out, const Actions& actions) {
    out << actions.v.size() << '\n';
    for (const Action action : actions.v)
        out << action << '\n';
    return out;
}

struct Solution {
    PathLength length;
    Actions actions;
};

Solution find_ans(const Instance& instance) {
    enum PointType {
        START, FINISH, KEY, LOCK
    };
    struct InterestingPoint {
        Coordinate x;
        PointType type;
        Index index;
        bool operator<(const InterestingPoint other) const {
            return x < other.x;
        }
    };
    const Index n = instance.key_locks.size();
    const Index INTEREST = 2 * n + 2;
    std::vector<InterestingPoint> interesting_points(INTEREST);
    interesting_points[0] = {.x = instance.start, .type = START, .index = 0};
    interesting_points[1] = {.x = instance.finish, .type = FINISH, .index = 0};
    for (Index i = 0; i < n; ++i) {
        interesting_points[2 * i + 2] = {.x = instance.key_locks[i].key, .type = KEY, .index = i};
        interesting_points[2 * i + 3] = {.x = instance.key_locks[i].lock, .type = LOCK, .index = i};
    }
    std::sort(interesting_points.begin(), interesting_points.end());
    Index start_index, finish_index;
    std::vector<Index> key_indices(n), lock_indices(n);
    for (Index i = 0; i < INTEREST; ++i)
        if (interesting_points[i].type == START)
            start_index = i;
        else if (interesting_points[i].type == FINISH)
            finish_index = i;
        else if (interesting_points[i].type == KEY)
            key_indices[interesting_points[i].index] = i;
        else
            lock_indices[interesting_points[i].index] = i;
    std::vector<Index> required_points(INTEREST);
    const Index IMPOSSIBLE_INDEX = -1;
    required_points[start_index] = start_index;
    for (Index i = start_index + 1; i < INTEREST; ++i) {
        required_points[i] = required_points[i - 1];
        if (interesting_points[i].type == LOCK && required_points[i] != IMPOSSIBLE_INDEX) {
            Index key_index = key_indices[interesting_points[i].index];
            required_points[i] = (key_index > i) ?
                                 IMPOSSIBLE_INDEX :
                                 std::min(required_points[i], key_index);
        }
    }
    for (Index i = start_index - 1; i + 1 > 0; --i) {
        required_points[i] = required_points[i + 1];
        if (interesting_points[i].type == LOCK && required_points[i] != IMPOSSIBLE_INDEX) {
            Index key_index = key_indices[interesting_points[i].index];
            required_points[i] = (key_index < i) ?
                                 IMPOSSIBLE_INDEX :
                                 std::max(required_points[i], key_index);
        }
    }
    std::vector<Index> destination{finish_index};
    while (destination.back() != start_index) {
        if (destination.back() == IMPOSSIBLE_INDEX)
            return {.length = IMPOSSIBLE_LENGTH, .actions = {}};
        destination.push_back(required_points[destination.back()]);
        if (destination.size() >= 3 && (
            destination.rbegin()[2] <= destination.rbegin()[0] && destination.rbegin()[0] > start_index
                ||
            destination.rbegin()[2] >= destination.rbegin()[0] && destination.rbegin()[0] < start_index
        ))
            return {.length = IMPOSSIBLE_LENGTH, .actions = {}};
    }
    const Index final_l_index = std::min(destination[0], destination[1]),
                final_r_index = std::max(destination[0], destination[1]);
    destination.pop_back();
    std::reverse(destination.begin(), destination.end());
    Index L = start_index, R = start_index;
    Solution solution{.length = 0, .actions = {}};
    for (Index i = final_l_index; i <= final_r_index; ++i)
        if (interesting_points[i].type == LOCK)
            solution.length += TIME_FOR_PAIR_OF_KEY_AND_LOCK;
    for (Index d : destination) {
        using SignedIndex = std::make_signed_t<Index>;
        SignedIndex direction = d > start_index ? 1 : -1;
        Index previous_stop = direction == 1 ? L : R;
        Index& current_start = direction == 1 ? R : L;
        solution.length += direction * (interesting_points[d].x - interesting_points[previous_stop].x);
        // below part is only needed for answer restoration
        if constexpr (NEED_ANSWER_RESTORATION) {
            Index current_stop = d + direction;
            for (Index i = current_start; i != current_stop; i += direction) {
                PointType point_type = interesting_points[i].type;
                bool need_go = point_type == FINISH || point_type == LOCK;
                if (point_type == KEY) {
                    Index lock = lock_indices[interesting_points[i].index];
                    need_go = final_l_index <= lock &&
                            final_r_index >= lock;
                }
                if (need_go) {
                    solution.actions.v.push_back({.type = GO, .x = interesting_points[i].x});
                    if (point_type == LOCK)
                        solution.actions.v.push_back({.type = UNLOCK, .x = interesting_points[i].x});
                    if (point_type == KEY)
                        solution.actions.v.push_back({.type = PICK, .x = interesting_points[i].x});
                }
            }
        }
        current_start = d;
    }
    return solution;
}

bool solve_test([[maybe_unused]] const Index test_index) {
    Instance instance;
    if (!(std::cin >> instance))
        return false;
    Solution solution = find_ans(instance);
    std::cout << solution.length << '\n';
    if constexpr (NEED_ANSWER_RESTORATION) {
        if (solution.length != IMPOSSIBLE_LENGTH)
            std::cerr << solution.actions << '\n';
    }
    return true;
}

void solve_tests() {
    Index n_tests;
    std::cin >> n_tests;
    for (Index test_index = 0; test_index < n_tests; ++test_index)
        if (!solve_test(test_index))
            return;
}

int main() {
    std::ios::sync_with_stdio(false);
    std::cin.tie(nullptr);
    solve_tests();
}
