import os
import csv
import pandas as pd
from sklearn.ensemble import RandomForestClassifier

# ---------------------------
# 1. Папка с файлами
# ---------------------------
BASE_DIR = os.path.dirname(os.path.abspath(__file__))
print(f"Работаем в папке: {BASE_DIR}")

# ---------------------------
# 2. Загрузка данных
# ---------------------------
try:
    X_train = pd.read_csv(os.path.join(BASE_DIR, "X_train.csv"))
    y_train = pd.read_csv(os.path.join(BASE_DIR, "y_train.csv"))
    X_test = pd.read_csv(os.path.join(BASE_DIR, "X_test.csv"))
    print(f"X_train: {X_train.shape}, y_train: {y_train.shape}, X_test: {X_test.shape}")
except Exception as e:
    print(f"Ошибка при загрузке файлов: {e}")
    exit(1)

# ---------------------------
# 3. Проверка названий колонок
# ---------------------------
print("X_train колонки:", X_train.columns)
print("X_test колонки:", X_test.columns)
print("y_train колонки:", y_train.columns)

# ---------------------------
# 4. Агрегация признаков по id
# ---------------------------
# Названия колонок с единицами измерения
SENSOR_COLUMNS = ['ax(g)','ay(g)','az(g)','wx(deg/s)','wy(deg/s)','wz(deg/s)']

def aggregate_features(df):
    features = df.groupby('id').agg({col: ['mean','std','min','max'] for col in SENSOR_COLUMNS})
    features.columns = ['_'.join(col) for col in features.columns]
    return features

X_train_agg = aggregate_features(X_train)
X_test_agg = aggregate_features(X_test)

# ---------------------------
# 5. Метки для обучающей выборки
# ---------------------------
y_train_agg = y_train.set_index('id')['label']

print(f"Агрегированные признаки: X_train {X_train_agg.shape}, X_test {X_test_agg.shape}")

# ---------------------------
# 6. Обучение модели
# ---------------------------
model = RandomForestClassifier(
    n_estimators=300,
    random_state=42,
    class_weight='balanced'
)
model.fit(X_train_agg, y_train_agg)
print("Модель обучена")

# ---------------------------
# 7. Предсказание
# ---------------------------
y_pred = model.predict(X_test_agg)
test_ids = X_test_agg.index

# ---------------------------
# 8. Создание solution.csv
# ---------------------------
solution_path = os.path.join(BASE_DIR, "solution.csv")
with open(solution_path, 'w', newline='') as f:
    writer = csv.writer(f)
    writer.writerow(['id','label'])
    for i, label in zip(test_ids, y_pred):
        writer.writerow([i,label])

print(f"Файл solution.csv успешно создан!\nПуть: {solution_path}")
