import sys
import math

# Константа для модуля
MOD = 998244353
# Максимальное значение A_i (используется только для справки)
MAX_A = 10**7

# --- Математические функции ---

def gcd(a, b):
    # Встроенный math.gcd (работает корректно с 0)
    return math.gcd(a, b)

def _lcm(a, b):
    """
    Вычисляет НОК без обрезания. 
    Используется для внутренней логики группировки, так как НОК может превышать MAX_A.
    """
    if a == 0 or b == 0:
        return 0
    # Используем стандартную формулу с учетом больших чисел
    # Примечание: Python автоматически обрабатывает большие целые числа
    return abs(a * b) // gcd(a, b)

# --- Решатель ---

def solve():
    try:
        # Ускорение чтения
        input = sys.stdin.read
        data = input().split()
        
        if not data: return
        
        N = int(data[0])
        # A - 1-индексация. A[0] не используется.
        A = [0] + [int(x) for x in data[1:N+1]] 
    except Exception:
        return

    # 1. Предобработка (Префиксный и Суффиксный GCD)
    P = [0] * (N + 2) # P[i] = gcd(A[1]...A[i])
    S = [0] * (N + 2) # S[i] = gcd(A[i]...A[N])
    
    # Расчет P: P[0]=0, gcd(0, x)=x
    for i in range(1, N + 1):
        P[i] = gcd(P[i-1], A[i]) if P[i-1] != 0 else A[i]
        
    # Расчет S: S[N+1]=0, gcd(0, x)=x
    for i in range(N, 0, -1):
        S[i] = gcd(S[i+1], A[i]) if S[i+1] != 0 else A[i]

    total_sum = 0
    
    # 2. Главный цикл по i (начало подмассива LCM: A[i..])
    for i in range(1, N + 1):
        
        # G_prefix = gcd(A[1]...A[i-1]). Используем P[i-1].
        G_prefix = P[i-1]
        
        j = i
        L_cur = 1 # Инициализируем LCM(A[i..j-1])
        
        # 3. Внешний цикл: Группировка по LCM (L_cur)
        # LCM(A[i..k]) меняется не более O(log A_max) раз
        while j <= N:
            j_start = j
            L_prev = L_cur
            
            # Обновляем L_cur для нового блока A[i..j_start]
            # L_cur = LCM(A[i..j_start])
            L_cur = _lcm(L_prev, A[j_start])
            
            # --- Находим конец блока j_end, где LCM(A[i..k]) остается L_cur ---
            j_end = j_start
            temp_j = j_start + 1
            while temp_j <= N:
                next_lcm = _lcm(L_cur, A[temp_j])
                
                # Ищем момент, когда LCM изменится
                if next_lcm != L_cur:
                    break
                temp_j += 1
            j_end = temp_j - 1
            
            # -------------------------------------------------------------------
            
            # G_prime = gcd(G_prefix, L_cur)
            G_prime = gcd(G_prefix, L_cur)
            
            # 4. Внутренний цикл: Группировка по Суффиксному GCD (S_{k+1})
            # Suffix GCD меняется не более O(log A_max) раз
            k = j_start # k - индекс, до которого берется LCM (A[i..k])
            while k <= j_end:
                k_start = k
                
                # S_cur = gcd(A[k+1]...A[N]). S[k+1] - Суффиксный GCD.
                S_cur = S[k_start + 1] 
                
                # Находим конец подблока k_end, где S_{m+1} остается S_cur
                k_end_limit = j_end
                
                temp_k = k_start + 1
                # Ищем следующий индекс m, где S[m+1] != S_cur
                while temp_k <= k_end_limit and S[temp_k + 1] == S_cur:
                    temp_k += 1
                k_end = temp_k - 1
                
                # Количество элементов k в этом подблоке [k_start, k_end]
                count = k_end - k_start + 1
                
                # Боевая сила: f_val = gcd(G_prime, S_cur)
                f_val = gcd(G_prime, S_cur)
                
                # Добавление к сумме: count * f_val
                total_sum = (total_sum + count * f_val) % MOD
                
                k = k_end + 1
            
            j = j_end + 1

    # Вывод
    print(total_sum)

# Установка рекурсивных функций для встроенных math (для Python < 3.5)
if not hasattr(math, 'gcd'):
    def _gcd(a, b):
        while b:
            a, b = b, a % b
        return a
    math.gcd = _gcd

if __name__ == "__main__":
    solve()
