import sys
import math

# Константы для точности и PI
PI = math.pi
# Уменьшаем количество итераций (100^3 = 1,000,000 вызовов - должно быть достаточно)
ITERATIONS = 100
PRECISION = 1e-10

def calculate_area(R: float, tA: float, tB: float, tC: float) -> float:
    """
    Вычисляет площадь треугольника, вершины которого заданы угловыми координатами 
    (R, tA), (R, tB), (R, tC).
    
    Используется формула для тройного вектора-произведения (signed area) /
    суммы площадей трех треугольников, образованных центром окружности и вершинами.
    
    S = 0.5 * R^2 * (sin(tB - tA) + sin(tC - tB) + sin(tA - tC))
    """
    # math.sin(x + 2 * PI) == math.sin(x), поэтому + 2*PI не нужен
    term1 = math.sin(tB - tA)
    term2 = math.sin(tC - tB)
    term3 = math.sin(tA - tC) 
    
    # Мы максимизируем эту signed area; максимальное значение будет соответствовать
    # максимальной абсолютной площади.
    return 0.5 * R * R * (term1 + term2 + term3)

def optimize_B_and_C(R: float, tA: float, b1: float, b2: float, c1: float, c2: float) -> float:
    """
    Находит максимальную площадь для фиксированного tA, оптимизируя tB и tC 
    с помощью вложенного тернарного поиска.
    """
    
    def find_max_S_for_tC(tB_fixed: float) -> float:
        """Внутренний тернарный поиск для tC при фиксированных tA и tB."""
        L = c1
        R_search = c2
        
        for _ in range(ITERATIONS):
            m1 = L + (R_search - L) / 3.0
            m2 = R_search - (R_search - L) / 3.0
            
            S1 = calculate_area(R, tA, tB_fixed, m1)
            S2 = calculate_area(R, tA, tB_fixed, m2)
            
            if S1 < S2:
                L = m1
            else:
                R_search = m2
        
        # Проверяем конечные точки и найденную точку для максимальной площади
        max_S = calculate_area(R, tA, tB_fixed, L)
        max_S = max(max_S, calculate_area(R, tA, tB_fixed, c1))
        max_S = max(max_S, calculate_area(R, tA, tB_fixed, c2))
        return max_S

    # Средний тернарный поиск для tB
    L = b1
    R_search = b2
    
    for _ in range(ITERATIONS):
        m1 = L + (R_search - L) / 3.0
        m2 = R_search - (R_search - L) / 3.0
        
        # Вычисляем максимальную площадь для каждого tB с помощью внутреннего поиска по tC
        S1 = find_max_S_for_tC(m1)
        S2 = find_max_S_for_tC(m2)
        
        if S1 < S2:
            L = m1
        else:
            R_search = m2
            
    # Проверяем конечные точки и найденную точку для максимальной площади
    max_S = find_max_S_for_tC(L)
    max_S = max(max_S, find_max_S_for_tC(b1))
    max_S = max(max_S, find_max_S_for_tC(b2))
    return max_S


def solve():
    # Ускорение чтения (построчно)
    try:
        R_line = sys.stdin.readline()
        if not R_line: return
        R = float(R_line)
        
        angles_line = sys.stdin.readline()
        if not angles_line: return
        angles = [float(x) for x in angles_line.split()]
        
    except Exception:
        # Обработка пустых или некорректных данных
        return

    # a1, a2, b1, b2, c1, c2
    a1, a2, b1, b2, c1, c2 = angles
    
    # Внешний тернарный поиск для tA
    L = a1
    R_search = a2
    
    for _ in range(ITERATIONS):
        m1 = L + (R_search - L) / 3.0
        m2 = R_search - (R_search - L) / 3.0
        
        # Вычисляем максимальную площадь для каждого tA с помощью вложенного поиска
        S1 = optimize_B_and_C(R, m1, b1, b2, c1, c2)
        S2 = optimize_B_and_C(R, m2, b1, b2, c1, c2)
        
        if S1 < S2:
            L = m1
        else:
            R_search = m2

    # Финальный максимум, проверяя найденную точку и границы A
    max_S = optimize_B_and_C(R, L, b1, b2, c1, c2)
    max_S = max(max_S, optimize_B_and_C(R, a1, b1, b2, c1, c2))
    max_S = max(max_S, optimize_B_and_C(R, a2, b1, b2, c1, c2))

    # Форматирование вывода
    print(f"{max_S:.10f}")

if __name__ == "__main__":
    solve()
