#include "testlib.h"
#include "constants.h"

#include <bits/stdc++.h>

using namespace std;
using ll = long long;

template<typename T>
T get_opt(const string &key, const T &def) { return has_opt(key) ? opt<T>(key) : def; }

static inline int rndInt(int L, int R) { return rnd.wnext(L, R, 0); }

static inline long long rndLL(long long L, long long R) { return rnd.next(L, R); }

struct P {
    ll x, y;
};

static inline uint64_t splitmix64(uint64_t z) {
    z += 0x9e3779b97f4a7c15ULL;
    z = (z ^ (z >> 30)) * 0xbf58476d1ce4e5b9ULL;
    z = (z ^ (z >> 27)) * 0x94d049bb133111ebULL;
    return z ^ (z >> 31);
}

struct PH {
    size_t operator()(P const &p) const noexcept {
        uint64_t a = splitmix64((uint64_t) p.x);
        uint64_t b = splitmix64((uint64_t) p.y);
        return a ^ (b + 0x9e3779b97f4a7c15ULL + (a << 6) + (a >> 2));
    }
};

static inline bool operator==(P const &a, P const &b) { return a.x == b.x && a.y == b.y; }

template<class F>
static inline void fill_unique_points(int n, vector<ll> &xs, vector<ll> &ys, F gen) {
    xs.assign(n, 0);
    ys.assign(n, 0);
    unordered_set<P, PH> used;
    used.reserve(n * 2 + 7);
    for (int i = 0; i < n; ++i) {
        ll x = 0, y = 0;
        int at = 0;
        while (true) {
            gen(i, x, y);
            if (at > 20) {
                x += rndLL(-1000000LL, 1000000LL);
                y += rndLL(-1000000LL, 1000000LL);
            }
            if (used.insert({x, y}).second) {
                xs[i] = x;
                ys[i] = y;
                break;
            }
            ++at;
        }
    }
}

struct TestCase {
    int n, m, q;
    vector<ll> xs, ys;
    vector<pair < int, int>> edges;
    vector<ll> cuts;
};

static inline void
add_edge_unique(int u, int v, int n, vector<pair < int, int>>

&edges,
unordered_set<unsigned long long> &used
) {
if (u == v) return;
int a = min(u, v), b = max(u, v);
unsigned long long key = (unsigned long long) a * (unsigned long long) n + (unsigned long long) b;
if (used.
insert(key)
.second) edges.
emplace_back(a, b
);
}

static inline vector<pair < int, int>>

make_unique_random_edges(int n, int m) {
    vector<pair < int, int>>
    edges;
    edges.reserve(m);
    unordered_set<unsigned long long> used;
    used.reserve(m * 2 + 7);
    while ((int) edges.size() < m) { add_edge_unique(rndInt(0, n - 1), rndInt(0, n - 1), n, edges, used); }
    return edges;
}

static inline vector<ll> mix_queries(int q, const vector<ll> &xs) {
    vector<ll> res;
    res.reserve(q);
    if (xs.empty()) {
        for (int i = 0; i < q; ++i) res.push_back(rndLL(-constants::MAX_COORD, constants::MAX_COORD));
        return res;
    }
    ll mn = *min_element(xs.begin(), xs.end()), mx = *max_element(xs.begin(), xs.end());
    for (int i = 0; i < q; ++i) {
        int pick = rndInt(1, 100);
        if (pick <= 35) res.push_back(xs[rndInt(0, (int) xs.size() - 1)]);
        else if (pick <= 50) res.push_back(rndInt(0, 1) ? mn - 1 : mx + 1);
        else if (pick <= 65) {
            static const ll pool[6] = {-constants::MAX_COORD, -1, 0, 1, constants::MAX_COORD, 42};
            res.push_back(pool[rndInt(0, 5)]);
        } else res.push_back(rndLL(-constants::MAX_COORD, constants::MAX_COORD));
    }
    return res;
}

static inline vector<ll> repeated_queries(int q, const vector<ll> &base) {
    vector<ll> vals = base;
    if (vals.empty()) vals.push_back(0);
    vector<ll> res;
    res.reserve(q);
    for (int i = 0; i < q; ++i) res.push_back(vals[rndInt(0, (int) vals.size() - 1)]);
    return res;
}

static inline void print_file(const vector<TestCase> &cases) {
    cout << cases.size() << "\n";
    for (const auto &tc: cases) {
        cout << tc.n << ' ' << tc.m << ' ' << tc.q << "\n";
        for (int i = 0; i < tc.n; ++i) {
            if (i) cout << ' ';
            cout << tc.xs[i];
        }
        cout << "\n";
        for (int i = 0; i < tc.n; ++i) {
            if (i) cout << ' ';
            cout << tc.ys[i];
        }
        cout << "\n";
        for (auto [u, v]: tc.edges) cout << (u + 1) << ' ' << (v + 1) << "\n";
        for (int i = 0; i < tc.q; ++i) {
            if (i) cout << ' ';
            cout << tc.cuts[i];
        }
        cout << "\n";
    }
}

vector<TestCase> gen_small_pack() {
    const int T = 5000;
    long long remN = constants::SUM_N_LIMIT, remM = constants::SUM_M_LIMIT, remQ = constants::SUM_Q_LIMIT;
    vector<TestCase> cases;
    cases.reserve(T);
    for (int i = 0; i < T; ++i) {
        int left = T - i - 1;
        int maxN = (int) min<long long>(10, remN - left * 1);
        int n = rndInt(1, max(1, maxN));
        int maxMdeg = 8;
        int maxM = (int) min<long long>((long long) min(maxMdeg, n * (n - 1) / 2), remM);
        int m = (maxM > 0 ? rndInt(0, maxM) : 0);
        int maxQ = (int) min<long long>(20, remQ - left * 1);
        if (maxQ < 1) maxQ = 1;
        int q = rndInt(1, maxQ);

        vector<ll> xs, ys;
        fill_unique_points(n, xs, ys, [&](int, ll &x, ll &y) {
            x = rndLL(-1000, 1000);
            y = rndLL(-1000, 1000);
        });
        auto edges = make_unique_random_edges(n, m);
        auto cuts = mix_queries(q, xs);

        remN -= n;
        remM -= m;
        remQ -= q;
        cases.push_back({n, m, q, xs, ys, edges, cuts});
    }
    return cases;
}

vector<TestCase> gen_medium_pack() {
    long long remN = constants::SUM_N_LIMIT, remM = constants::SUM_M_LIMIT, remQ = constants::SUM_Q_LIMIT;
    vector<TestCase> cases;
    while (true) {
        if (remN <= 0 || remQ <= 0) break;
        int n_max = (int) min<long long>(1500, remN);
        int n_min = min<int>(50, (int) remN);
        int n = rndInt(max(1, n_min), max(1, n_max));

        int m_hi = min<int>((int) min<long long>(2LL * n, remM), n * (n - 1) / 2);
        int m_lo = min(n - 1, m_hi);
        int m = (m_hi > 0 ? rndInt(m_lo, m_hi) : 0);

        int q_hi = (int) min<long long>((long long) max(1, 2 * n), remQ);
        int q_lo = min<int>(max(1, n / 2), q_hi);
        int q = rndInt(q_lo, q_hi);

        if (remN - n < 0 || remM - m < 0 || remQ - q < 0) break;

        vector<ll> xs, ys;
        fill_unique_points(n, xs, ys, [&](int, ll &x, ll &y) {
            x = rndLL(-2000000, 2000000);
            y = rndLL(-2000000, 2000000);
        });
        auto edges = make_unique_random_edges(n, m);
        auto cuts = mix_queries(q, xs);

        remN -= n;
        remM -= m;
        remQ -= q;
        cases.push_back({n, m, q, xs, ys, edges, cuts});

        if ((int) cases.size() > 3000) break;
        if (remN < 50 && remQ < 50) break;
    }
    if (cases.empty()) {
        int n = 50, m = 49, q = 50;
        vector<ll> xs, ys;
        fill_unique_points(n, xs, ys, [&](int i, ll &x, ll &y) {
            x = i;
            y = 0;
        });
        vector<pair < int, int>>
        edges;
        for (int i = 1; i < n; ++i) edges.emplace_back(i - 1, i);
        auto cuts = mix_queries(q, xs);
        cases.push_back({n, (int) edges.size(), q, xs, ys, edges, cuts});
    }
    return cases;
}

TestCase big_random_sparse() {
    int n = constants::SUM_N_LIMIT, m = constants::SUM_M_LIMIT, q = constants::SUM_Q_LIMIT;
    vector<ll> xs, ys;
    fill_unique_points(n, xs, ys, [&](int, ll &x, ll &y) {
        x = rndLL(-constants::MAX_COORD, constants::MAX_COORD);
        y = rndLL(-constants::MAX_COORD, constants::MAX_COORD);
    });
    auto edges = make_unique_random_edges(n, m);
    auto cuts = mix_queries(q, xs);
    return {n, m, q, xs, ys, edges, cuts};
}

TestCase big_random_dense_smalln() {
    int n = 1500;
    int m = min<int>(constants::SUM_M_LIMIT, n * (n - 1) / 2);
    int q = constants::SUM_Q_LIMIT;
    vector<ll> xs, ys;
    fill_unique_points(n, xs, ys, [&](int, ll &x, ll &y) {
        x = rndLL(-10000000LL, 10000000LL);
        y = rndLL(-10000000LL, 10000000LL);
    });
    auto edges = make_unique_random_edges(n, m);
    auto cuts = mix_queries(q, xs);
    return {n, m, q, xs, ys, edges, cuts};
}

TestCase big_tree() {
    int n = constants::SUM_N_LIMIT;
    int m = n - 1;
    int q = constants::SUM_Q_LIMIT;
    vector<ll> xs, ys;
    fill_unique_points(n, xs, ys, [&](int, ll &x, ll &y) {
        x = rndLL(-1000000000LL, 1000000000LL);
        y = rndLL(-1000000000LL, 1000000000LL);
    });
    vector<pair < int, int>>
    edges;
    edges.reserve(m);
    for (int v = 1; v < n; ++v) {
        int u = rndInt(0, v - 1);
        edges.emplace_back(u, v);
    }
    auto cuts = mix_queries(q, xs);
    return {n, m, q, xs, ys, edges, cuts};
}

TestCase big_path_sorted_x() {
    int n = constants::SUM_N_LIMIT, m = n - 1, q = constants::SUM_Q_LIMIT;
    vector<ll> xs, ys;
    fill_unique_points(n, xs, ys, [&](int i, ll &x, ll &y) {
        x = (ll) i * 10;
        y = rndLL(-10, 10);
    });
    vector<pair < int, int>>
    edges;
    edges.reserve(m);
    for (int i = 1; i < n; ++i) edges.emplace_back(i - 1, i);
    vector<ll> cuts;
    cuts.reserve(q);
    for (int i = 0; i < q; ++i) {
        if (i & 1) cuts.push_back(xs[rndInt(0, n - 1)]);
        else
            cuts.push_back(rndLL(-1, (ll) 10 * n + 1));
    }
    return {n, m, q, xs, ys, edges, cuts};
}

TestCase big_star_center0() {
    int n = constants::SUM_N_LIMIT, m = n - 1, q = constants::SUM_Q_LIMIT;
    vector<ll> xs, ys;
    fill_unique_points(n, xs, ys, [&](int i, ll &x, ll &y) {
        if (i == 0) {
            x = 0;
            y = 0;
        } else {
            x = rndLL(-1000000000LL, 1000000000LL);
            y = rndLL(-100000LL, 100000LL);
        }
    });
    vector<pair < int, int>>
    edges;
    edges.reserve(m);
    for (int i = 1; i < n; ++i) edges.emplace_back(0, i);
    auto cuts = repeated_queries(q, {-1, 0, 1, -10, 10});
    return {n, m, q, xs, ys, edges, cuts};
}

TestCase big_two_clusters_many_cross() {
    int n = constants::SUM_N_LIMIT, q = constants::SUM_Q_LIMIT;
    int A = n / 2, B = n - A;
    vector<ll> xs, ys;
    fill_unique_points(n, xs, ys, [&](int i, ll &x, ll &y) {
        if (i < A) {
            x = -1000000LL - i;
            y = rndLL(-1000000LL, 1000000LL);
        } else {
            x = 1000000LL + (i - A);
            y = rndLL(-1000000LL, 1000000LL);
        }
    });
    vector<pair < int, int>>
    edges;
    edges.reserve(constants::SUM_M_LIMIT);
    unordered_set<unsigned long long> used;
    used.reserve(constants::SUM_M_LIMIT * 2 + 7);
    while ((int) edges.size() < constants::SUM_M_LIMIT)
        add_edge_unique(rndInt(0, A - 1), rndInt(A, n - 1), n, edges, used);
    auto cuts = repeated_queries(q, {-1, 0, 1});
    return {n, (int) edges.size(), q, xs, ys, edges, cuts};
}

TestCase big_all_x_same() {
    int n = constants::SUM_N_LIMIT, m = constants::SUM_M_LIMIT, q = constants::SUM_Q_LIMIT;
    vector<ll> xs, ys;
    fill_unique_points(n, xs, ys, [&](int, ll &x, ll &y) {
        x = 0;
        y = rndLL(-constants::MAX_COORD, constants::MAX_COORD);
    });
    auto edges = make_unique_random_edges(n, m);
    vector<ll> cuts(q, 0);
    for (int i = 0; i < q; ++i) if (i % 3) cuts[i] = rndLL(-constants::MAX_COORD, constants::MAX_COORD);
    return {n, m, q, xs, ys, edges, cuts};
}

TestCase big_all_y_same() {
    int n = constants::SUM_N_LIMIT, m = constants::SUM_M_LIMIT, q = constants::SUM_Q_LIMIT;
    vector<ll> xs, ys;
    fill_unique_points(n, xs, ys, [&](int, ll &x, ll &y) {
        x = rndLL(-constants::MAX_COORD, constants::MAX_COORD);
        y = 123456789LL;
    });
    auto edges = make_unique_random_edges(n, m);
    auto cuts = mix_queries(q, xs);
    return {n, m, q, xs, ys, edges, cuts};
}

TestCase big_zero_edges() {
    int n = constants::SUM_N_LIMIT, m = 0, q = constants::SUM_Q_LIMIT;
    vector<ll> xs, ys;
    fill_unique_points(n, xs, ys, [&](int, ll &x, ll &y) {
        x = rndLL(-1000000000LL, 1000000000LL);
        y = rndLL(-1000000000LL, 1000000000LL);
    });
    vector<pair < int, int>>
    edges;
    auto cuts = mix_queries(q, xs);
    return {n, m, q, xs, ys, edges, cuts};
}

TestCase big_grid_224x224() {
    int R = 224, C = 224;
    int n = R * C;
    vector<ll> xs(n), ys(n);
    auto id = [&](int r, int c) { return r * C + c; };
    for (int r = 0; r < R; ++r)
        for (int c = 0; c < C; ++c) {
            int v = id(r, c);
            xs[v] = c;
            ys[v] = r;
        }
    vector<pair < int, int>>
    edges;
    edges.reserve(2 * n);
    for (int r = 0; r < R; ++r)
        for (int c = 0; c < C; ++c) {
            int v = id(r, c);
            if (c + 1 < C) edges.emplace_back(v, id(r, c + 1));
            if (r + 1 < R) edges.emplace_back(v, id(r + 1, c));
        }
    int m = (int) edges.size();
    int q = constants::SUM_Q_LIMIT;
    auto cuts = mix_queries(q, xs);
    return {n, m, q, xs, ys, edges, cuts};
}

TestCase big_many_equal_queries() {
    int n = constants::SUM_N_LIMIT, m = constants::SUM_M_LIMIT, q = constants::SUM_Q_LIMIT;
    vector<ll> xs, ys;
    fill_unique_points(n, xs, ys, [&](int i, ll &x, ll &y) {
        x = (i % 3) - 1;
        y = rndLL(-constants::MAX_COORD, constants::MAX_COORD);
    });
    auto edges = make_unique_random_edges(n, m);
    auto cuts = repeated_queries(q, {-1, 0, 1});
    return {n, m, q, xs, ys, edges, cuts};
}

TestCase big_extreme_coords() {
    int n = constants::SUM_N_LIMIT, m = constants::SUM_M_LIMIT, q = constants::SUM_Q_LIMIT;
    vector<ll> xs, ys;
    fill_unique_points(n, xs, ys, [&](int i, ll &x, ll &y) {
        x = (i & 1) ? constants::MAX_COORD : -constants::MAX_COORD;
        y = rndLL(-constants::MAX_COORD, constants::MAX_COORD);
    });
    auto edges = make_unique_random_edges(n, m);
    vector<ll> cuts;
    cuts.reserve(q);
    for (int i = 0; i < q; ++i) {
        int t = rndInt(0, 4);
        if (t == 0) cuts.push_back(constants::MAX_COORD);
        else if (t == 1) cuts.push_back(-constants::MAX_COORD);
        else if (t == 2) cuts.push_back(constants::MAX_COORD - 1);
        else if (t == 3) cuts.push_back(-constants::MAX_COORD + 1);
        else cuts.push_back(rndLL(-constants::MAX_COORD, constants::MAX_COORD));
    }
    return {n, m, q, xs, ys, edges, cuts};
}

TestCase big_clique_strips() {
    int n = constants::SUM_N_LIMIT, targetM = constants::SUM_M_LIMIT;
    int groups = 32, per = max(1, n / groups);
    vector<ll> xs, ys;
    fill_unique_points(n, xs, ys, [&](int i, ll &x, ll &y) {
        int g = min(groups - 1, i / per);
        x = (ll) g * 10;
        y = rndLL(-1000000LL, 1000000LL);
    });
    vector<pair < int, int>>
    edges;
    unordered_set<unsigned long long> used;
    used.reserve(targetM * 2 + 7);
    int m = 0;
    for (int g = 0; g < groups && m < targetM; ++g) {
        int L = g * per, R = min(n, L + per);
        int want = min(targetM - m, (R - L) * (R - L - 1) / 2);
        while (want-- > 0 && m < targetM) {
            int u = rndInt(L, max(L, R - 1)), v = rndInt(L, max(L, R - 1));
            if (u == v) continue;
            add_edge_unique(u, v, n, edges, used);
            m = (int) edges.size();
        }
    }
    int q = constants::SUM_Q_LIMIT;
    auto cuts = mix_queries(q, xs);
    return {n, (int) edges.size(), q, xs, ys, edges, cuts};
}

vector<TestCase> build_by_preset(int preset) {
    if (1 <= preset && preset <= 5) return gen_small_pack();
    if (6 <= preset && preset <= 10) return gen_medium_pack();
    vector<TestCase> v;
    v.reserve(1);
    switch (preset) {
        case 11:
            v.push_back(big_random_sparse());
            break;
        case 12:
            v.push_back(big_random_dense_smalln());
            break;
        case 13:
            v.push_back(big_tree());
            break;
        case 14:
            v.push_back(big_path_sorted_x());
            break;
        case 15:
            v.push_back(big_star_center0());
            break;
        case 16:
            v.push_back(big_random_sparse());
            break;
        case 17:
            v.push_back(big_two_clusters_many_cross());
            break;
        case 18:
            v.push_back(big_all_x_same());
            break;
        case 19:
            v.push_back(big_all_y_same());
            break;
        case 20:
            v.push_back(big_zero_edges());
            break;
        case 21:
            v.push_back(big_grid_224x224());
            break;
        case 22:
            v.push_back(big_many_equal_queries());
            break;
        case 23:
            v.push_back(big_extreme_coords());
            break;
        case 24:
            v.push_back(big_clique_strips());
            break;
        default:
            v.push_back(big_random_sparse());
            break;
    }
    return v;
}

int main(int argc, char *argv[]) {
    registerGen(argc, argv, 1);
    int preset = get_opt<int>("preset", 1);
    auto cases = build_by_preset(preset);

    long long sN = 0, sM = 0, sQ = 0;
    for (auto &tc: cases) {
        ensuref(tc.n >= 1 && tc.m >= 0 && tc.q >= 1, "Bad n/m/q");
        ensuref(tc.m <= (long long) tc.n * (tc.n - 1) / 2, "Too many edges for n");
        ensuref((int) tc.xs.size() == tc.n && (int) tc.ys.size() == tc.n, "coords mismatch");
        ensuref((int) tc.edges.size() == tc.m, "edges size mismatch");
        ensuref((int) tc.cuts.size() == tc.q, "queries size mismatch");

        unordered_set<P, PH> seen;
        seen.reserve(tc.n * 2 + 7);
        for (int i = 0; i < tc.n; ++i) ensuref(seen.insert({tc.xs[i], tc.ys[i]}).second, "duplicate village coords");

        sN += tc.n;
        sM += tc.m;
        sQ += tc.q;
    }
    ensuref((int) cases.size() <= constants::MAX_T, "Too many testcases (t)");
    ensuref(sN <= constants::SUM_N_LIMIT, "Sum n limit exceeded");
    ensuref(sM <= constants::SUM_M_LIMIT, "Sum m limit exceeded");
    ensuref(sQ <= constants::SUM_Q_LIMIT, "Sum q limit exceeded");

    print_file(cases);
}
