#include "testlib.h"
#include "constants.h"
#include <bits/stdc++.h>

using namespace std;
using ll = long long;

/* ---------- structures ---------- */
struct Query {
    int k, r;
    vector<ll> a;
};
struct DataSet {
    int n, q;
    vector<ll> x;
    vector<Query> queries;
};

template<typename T>
T get_opt(const string &key, const T &def) { return has_opt(key) ? opt<T>(key) : def; }

template<class T>
static void shuffle_vec(vector<T> &v) {
    for (int i = 0; i < (int) v.size(); ++i) {
        int j = rnd.next(i, (int) v.size() - 1);
        if (i != j) swap(v[i], v[j]);
    }
}

/* ---------- modes ---------- */
enum Suite {
    SMALLRAND, RANDBIG, ONEPOINT, TWOPOINTS, ARITH, CLUSTERS, ALTGAPS,
    UNSORTED, ZEROCASE, MANYQ, STRESS, RKMIX
};

static Suite parseSuite(const string &s) {
    if (s == "smallrand") return SMALLRAND;
    if (s == "randbig") return RANDBIG;
    if (s == "onepoint") return ONEPOINT;
    if (s == "twopoints") return TWOPOINTS;
    if (s == "ap") return ARITH;
    if (s == "clusters") return CLUSTERS;
    if (s == "altgaps") return ALTGAPS;
    if (s == "unsorted") return UNSORTED;
    if (s == "zero") return ZEROCASE;
    if (s == "manyq") return MANYQ;
    if (s == "stress") return STRESS;
    if (s == "rkmix") return RKMIX;
    return SMALLRAND;
}

/* ---------- product-budget (Σ n*q) ---------- */
struct Budget {
    long long sum_nq_limit;
    long long sum_r_limit;
    long long used_nq_prod = 0;
    long long used_r = 0;

    inline long long remProd() const { return sum_nq_limit - used_nq_prod; }

    bool canStartDS(int n) const { return n >= 1 && used_nq_prod + (long long) n <= sum_nq_limit; }

    bool canAddOneQuery(int n, int q_cur, int rsum_cur, int r_add) const {
        if (n <= 0) return false;
        if (used_r + rsum_cur + r_add > sum_r_limit) return false;
        long long need_prod = (long long) n * (long long) (q_cur + 1);
        return used_nq_prod + need_prod <= sum_nq_limit;
    }

    void addDS(int n, int q, int rsum) {
        used_nq_prod += (long long) n * (long long) q;
        used_r += rsum;
    }
};

/* ---------- helpers ---------- */
static inline ll clamp_coord(ll v, ll lim) {
    if (v < -lim) return -lim;
    if (v > lim) return lim;
    return v;
}

static pair<ll, ll> minmax_vec(const vector<ll> &v) {
    if (v.empty()) return {0, 0};
    ll mn = (ll) 4e18, mx = -(ll) 4e18;
    for (ll x: v) {
        mn = min(mn, x);
        mx = max(mx, x);
    }
    return {mn, mx};
}

/* ---------- people generators ---------- */
static vector<ll> people_random(int n, ll lim) {
    vector<ll> x(n);
    for (int i = 0; i < n; i++) x[i] = rnd.next(-lim, lim);
    return x;
}

static vector<ll> people_one_point(int n, ll v) {
    vector<ll> x(n, clamp_coord(v, constants::MAX_COORD));
    return x;
}

static vector<ll> people_two_points(int n, ll a, ll b) {
    vector<ll> x(n);
    for (int i = 0; i < n; i++) x[i] = (i % 2 == 0 ? a : b);
    shuffle_vec(x);
    return x;
}

static vector<ll> people_arith(int n, ll start, ll step, ll lim) {
    vector<ll> x(n);
    if (n == 0) return x;
    x[0] = clamp_coord(start, lim);
    for (int i = 1; i < n; i++) {
        long double cand = (long double) x[i - 1] + (long double) step;
        if (cand < -(long double) lim) cand = -(long double) lim;
        if (cand > (long double) lim) cand = (long double) lim;
        x[i] = (ll) cand;
    }
    return x;
}

static vector<ll> people_clusters(int n, int clusters, ll lim) {
    clusters = max(1, min(clusters, n));
    vector<int> sz(clusters, 0);
    for (int i = 0; i < n; i++) sz[rnd.next(0, clusters - 1)]++;
    vector<ll> centers(clusters);
    for (int i = 0; i < clusters; i++) {
        long double rel =
                -(long double) lim + (long double) (2LL * i + 1) * ((long double) lim / (long double) clusters);
        centers[i] = (ll) floor(rel);
    }
    vector<ll> x;
    x.reserve(n);
    for (int c = 0; c < clusters; c++)
        for (int j = 0; j < sz[c]; j++) {
            ll jitter = rnd.next(-1000, 1000);
            x.push_back(clamp_coord(centers[c] + jitter, lim));
        }
    shuffle_vec(x);
    return x;
}

static vector<ll> people_altgaps(int n, ll bigL, ll smallS, ll start, ll lim) {
    vector<ll> x(n);
    if (n == 0) return x;
    x[0] = clamp_coord(start, lim);
    for (int i = 1; i < n; i++) {
        ll d = (i % 2 ? bigL : smallS);
        long double cand = (long double) x[i - 1] + (long double) d;
        if (cand < -(long double) lim) cand = -(long double) lim;
        if (cand > (long double) lim) cand = (long double) lim;
        x[i] = (ll) cand;
    }
    return x;
}

/* ---------- fixed centers ---------- */
static vector<ll> random_fixed_centers(int r, const vector<ll> &x, ll lim, bool biasOutside = false) {
    vector<ll> a;
    a.reserve(r);
    auto [mn, mx] = minmax_vec(x);
    for (int i = 0; i < r; i++) {
        int typ = rnd.next(0, 3);
        if (biasOutside && rnd.next(0, 1)) typ = 3;
        ll v = 0;
        if (typ == 0 && !x.empty()) {
            v = x[rnd.next(0, (int) x.size() - 1)];
        } else if (typ == 1) {
            ll L = mn - 5, R = mx + 5;
            if (L > R) swap(L, R);
            v = rnd.next(L, R);
        } else if (typ == 2 && (int) x.size() >= 2) {
            int i1 = rnd.next(0, (int) x.size() - 1);
            int j1 = rnd.next(0, (int) x.size() - 1);
            if (i1 == j1) j1 = (j1 + 1) % (int) x.size();
            long double mid = ((long double) x[i1] + (long double) x[j1]) / 2.0L;
            v = (ll) floor(mid);
        } else {
            ll side = rnd.next(0, 1) ? 1 : -1;
            ll delta = rnd.next(lim / 2, lim);
            v = (side == 1 ? mx + delta : mn - delta);
        }
        a.push_back(clamp_coord(v, lim));
    }
    return a;
}

/* ---------- queries ---------- */
static Query make_query_general(const vector<ll> &x, ll lim, int r_budget_left) {
    int n = (int) x.size();
    int k_hi = min(constants::MAX_K, n + max(10, n * 3));
    int k = rnd.next(1, max(1, k_hi));
    int r = rnd.next(0, k);
    r = min(r, max(0, r_budget_left));
    int coin = rnd.next(1, 100);
    if (coin <= 20) r = 0;
    else if (coin <= 40) r = min(k, max(0, r_budget_left));
    auto a = random_fixed_centers(r, x, lim, rnd.next(0, 3) == 0);
    return Query{k, r, a};
}

static Query make_query_rk(const vector<ll> &x, ll lim, int r_budget_left) {
    int n = (int) x.size();
    int k_hi = min(constants::MAX_K, n + max(10, n * 2));
    int k = rnd.next(1, max(1, k_hi));
    int r = min(k, max(0, r_budget_left));
    if (r == 0) return make_query_general(x, lim, r_budget_left);
    auto a = random_fixed_centers(r, x, lim, true);
    return Query{k, r, a};
}

static Query make_query_r0(const vector<ll> &x, ll lim) {
    int n = (int) x.size();
    int k_hi = min(constants::MAX_K, n + max(10, n * 3));
    int k = rnd.next(1, max(1, k_hi));
    return Query{k, 0, {}};
}

static Query make_query_zero_easy(const vector<ll> &x, ll /*lim*/) {
    int n = (int) x.size();
    int k = max(n, rnd.next(n, min(constants::MAX_K, n + 10 + max(10, n * 2))));
    return Query{k, 0, {}};
}

static Query make_query_fixed_exactXi(const vector<ll> &x, int take, ll lim, int r_budget_left) {
    vector<ll> pool = x;
    if (pool.empty()) return make_query_zero_easy(x, lim);
    shuffle_vec(pool);
    take = max(0, min(take, (int) pool.size()));
    take = min(take, max(0, r_budget_left));
    vector<ll> a;
    a.reserve(take);
    for (int i = 0; i < take; i++) a.push_back(pool[i]);
    int n = (int) x.size();
    int k_hi = min(constants::MAX_K, n + max(10, n * 2));
    int k = rnd.next(max(1, take), max(1, k_hi));
    int r = take;
    return Query{k, r, a};
}

/* ---------- print ---------- */
static void print_all(const vector<DataSet> &sets) {
    cout << (int) sets.size() << "\n";
    for (const auto &ds: sets) {
        cout << ds.n << " " << ds.q << "\n";
        for (int i = 0; i < ds.n; i++) {
            if (i) cout << ' ';
            cout << ds.x[i];
        }
        cout << "\n";
        for (const auto &qq: ds.queries) {
            cout << qq.k << " " << qq.r;
            for (int i = 0; i < qq.r; i++) cout << " " << qq.a[i];
            cout << "\n";
        }
    }
}

/* ---------- dataset builder (product-aware) ---------- */
static DataSet make_dataset_generic(const vector<ll> &people, int q_goal_hint, Budget &budget, ll coord_lim,
                                    int pattern /*0: general, 1:r0, 2:rk, 3:zero*/ ) {
    DataSet empty;
    empty.n = 0;
    empty.q = 0;

    if (!budget.canStartDS((int) people.size())) return empty;

    long long rem = budget.remProd();
    int n = (int) people.size();
    long long q_cap = max(1LL, rem / max(1, n));

    long long q_goal = min<long long>(q_goal_hint, q_cap);

    DataSet ds;
    ds.n = n;
    ds.x = people;
    ds.q = 0;
    vector<Query> qs;
    qs.reserve((int) min<long long>(q_goal, 100000LL));
    int rsum = 0;

    for (long long it = 0; it < q_goal; ++it) {
        Query q;
        if (pattern == 1) q = make_query_r0(people, coord_lim);
        else if (pattern == 2)
            q = make_query_rk(people, coord_lim, max(0, (int) (budget.sum_r_limit - budget.used_r - rsum)));
        else if (pattern == 3) {
            if (rnd.next(0, 1)) q = make_query_zero_easy(people, coord_lim);
            else
                q = make_query_fixed_exactXi(people, min((int) people.size(), rnd.next(1, max(1, (int) people.size()))),
                                             coord_lim, max(0, (int) (budget.sum_r_limit - budget.used_r - rsum)));
        } else q = make_query_general(people, coord_lim, max(0, (int) (budget.sum_r_limit - budget.used_r - rsum)));

        if (!budget.canAddOneQuery(ds.n, (int) qs.size(), rsum, q.r)) break;
        rsum += q.r;
        qs.push_back(std::move(q));
    }

    if (qs.empty()) {
        Query q = make_query_r0(people, coord_lim);
        if (!budget.canAddOneQuery(ds.n, 0, 0, 0)) return empty;
        qs.push_back(q);
    }

    ds.q = (int) qs.size();
    ds.queries = std::move(qs);
    budget.addDS(ds.n, ds.q, rsum);
    return ds;
}

/* ---------- suite builder ---------- */
static vector<DataSet> build_suite(Suite suite, Budget budget, int want_t, ll coord_lim, int max_n_small) {
    vector<DataSet> out;
    out.reserve(want_t);
    auto push = [&](DataSet &&ds) { if (ds.n > 0 && ds.q > 0) out.push_back(std::move(ds)); };

    auto fill_many_small = [&]() {
        while ((int) out.size() < want_t && budget.remProd() > 0) {
            long long rem = budget.remProd();
            int n = rnd.next(1, min<int>(max_n_small, (int) rem));
            if (!budget.canStartDS(n)) break;

            vector<ll> ppl = people_random(n, min<ll>(coord_lim, 200));
            long long q_cap = max(1LL, rem / n);
            long long q_goal = min<long long>(q_cap, rnd.next(50, 1000));
            auto ds = make_dataset_generic(ppl, (int) q_goal, budget, coord_lim, 0);
            if (ds.n == 0) break;
            push(std::move(ds));
            if (budget.remProd() <= 0) break;
        }
    };

    switch (suite) {
        case SMALLRAND: {
            fill_many_small();
        }
            break;

        case RANDBIG: {
            int blocks = rnd.next(1, 3);
            for (int b = 0; b < blocks; b++) {
                long long rem = budget.remProd();
                if (rem <= 0) break;
                int n_try = rnd.next(1000, 60000);
                int n = min<int>(n_try, (int) rem);
                n = max(1, n);
                if (!budget.canStartDS(n)) break;

                vector<ll> ppl = people_random(n, coord_lim);
                long long q_cap = max(1LL, rem / n);
                long long q_goal = min<long long>(q_cap, 2000LL);
                auto ds = make_dataset_generic(ppl, (int) q_goal, budget, coord_lim, 0);
                if (ds.n == 0) break;
                push(std::move(ds));
                if (budget.remProd() <= 0) break;
            }
        }
            break;

        case ONEPOINT: {
            for (int b = 0; b < max(1, want_t); b++) {
                long long rem = budget.remProd();
                if (rem <= 0) break;
                int n = min<int>(rnd.next(1, 70000), (int) rem);
                if (!budget.canStartDS(n)) break;
                ll v = rnd.next(-constants::MAX_COORD / 2, constants::MAX_COORD / 2);
                auto ppl = people_one_point(n, v);
                long long q_cap = max(1LL, rem / n);
                auto ds = make_dataset_generic(ppl, (int) q_cap, budget, coord_lim, (rnd.next(0, 1) ? 3 : 0));
                push(std::move(ds));
                if (budget.remProd() <= 0) break;
            }
        }
            break;

        case TWOPOINTS: {
            for (int b = 0; b < max(1, want_t); b++) {
                long long rem = budget.remProd();
                if (rem <= 0) break;
                int n = min<int>(rnd.next(2, 70000), (int) rem);
                if (!budget.canStartDS(n)) break;
                ll a = rnd.next(-constants::MAX_COORD / 2, constants::MAX_COORD / 2);
                ll bcoord = clamp_coord(a + rnd.next((ll) 1, (ll) min<ll>(constants::MAX_COORD, 1000000000LL)),
                                        constants::MAX_COORD);
                auto ppl = people_two_points(n, a, bcoord);
                long long q_cap = max(1LL, rem / n);
                auto ds = make_dataset_generic(ppl, (int) q_cap, budget, coord_lim, 0);
                push(std::move(ds));
                if (budget.remProd() <= 0) break;
            }
        }
            break;

        case ARITH: {
            long long rem = budget.remProd();
            if (rem <= 0) break;
            int n = (int) min<long long>(80000LL, max(2LL, rem));
            ll step = max<ll>(1, (2 * coord_lim) / max(2, n) / 2);
            ll start = rnd.next(-constants::MAX_COORD / 2, constants::MAX_COORD / 2);
            auto ppl = people_arith(n, start, step, coord_lim);
            long long q_cap = max(1LL, rem / n);
            auto ds = make_dataset_generic(ppl, (int) min<long long>(q_cap, 2000LL), budget, coord_lim, 0);
            push(std::move(ds));
        }
            break;

        case CLUSTERS: {
            long long rem = budget.remProd();
            if (rem <= 0) break;
            int n = (int) min<long long>(70000LL, max(10LL, rem));
            int K = rnd.next(2, 5);
            auto ppl = people_clusters(n, K, coord_lim);
            long long q_cap = max(1LL, rem / n);
            auto ds = make_dataset_generic(ppl, (int) min<long long>(q_cap, 2000LL), budget, coord_lim, 0);
            push(std::move(ds));
        }
            break;

        case ALTGAPS: {
            for (int rep = 0; rep < max(1, want_t); rep++) {
                long long rem = budget.remProd();
                if (rem <= 0) break;
                int n = min<int>(rnd.next(1000, 8000), (int) rem);
                if (!budget.canStartDS(n)) break;
                ll bigL = rnd.next(200000, 1000000);
                ll smallS = rnd.next(1, 5);
                ll start = rnd.next(-constants::MAX_COORD / 2, constants::MAX_COORD / 2);
                auto ppl = people_altgaps(n, bigL, smallS, start, coord_lim);
                long long q_cap = max(1LL, rem / n);
                auto ds = make_dataset_generic(ppl, (int) min<long long>(q_cap, 10000LL), budget, coord_lim, 0);
                push(std::move(ds));
                if (budget.remProd() <= 0) break;
            }
        }
            break;

        case UNSORTED: {
            long long rem = budget.remProd();
            if (rem <= 0) break;
            int n_try = rnd.next(1000, 70000);
            int n = min<int>(n_try, (int) rem);
            if (!budget.canStartDS(n)) break;
            auto ppl = people_random(n, coord_lim);
            if (n >= 3) {
                ppl[0] = -constants::MAX_COORD;
                ppl[n - 1] = constants::MAX_COORD;
                ppl[n / 2] = 0;
            }
            long long q_cap = max(1LL, rem / n);
            auto ds = make_dataset_generic(ppl, (int) min<long long>(q_cap, 2000LL), budget, coord_lim, 0);
            push(std::move(ds));
        }
            break;

        case ZEROCASE: {
            long long rem = budget.remProd();
            if (rem <= 0) break;
            int n_try = rnd.next(1000, 40000);
            int n = min<int>(n_try, (int) rem);
            if (!budget.canStartDS(n)) break;
            auto ppl = people_random(n, min<ll>(coord_lim, 1000000));
            long long q_cap = max(1LL, rem / n);
            DataSet ds;
            ds.n = n;
            ds.x = ppl;
            int rsum = 0;
            long long q_goal = min<long long>(q_cap, 10000LL);
            for (long long i = 0; i < q_goal; i++) {
                Query q;
                int typ = rnd.next(1, 100);
                if (typ <= 60) q = make_query_zero_easy(ppl, coord_lim);
                else if (typ <= 85)
                    q = make_query_fixed_exactXi(ppl, rnd.next(1, min(20, n)), coord_lim,
                                                 max(0, (int) (budget.sum_r_limit - budget.used_r - rsum)));
                else q = make_query_general(ppl, coord_lim, max(0, (int) (budget.sum_r_limit - budget.used_r - rsum)));
                if (!budget.canAddOneQuery(n, (int) ds.queries.size(), rsum, q.r)) break;
                rsum += q.r;
                ds.queries.push_back(std::move(q));
            }
            if (ds.queries.empty()) ds.queries.push_back(make_query_zero_easy(ppl, coord_lim));
            ds.q = (int) ds.queries.size();
            budget.addDS(ds.n, ds.q, rsum);
            push(std::move(ds));
        }
            break;

        case MANYQ: {
            long long rem = budget.remProd();
            if (rem <= 0) break;
            int n = rnd.next(1, (int) min<long long>(50LL, rem));
            if (!budget.canStartDS(n)) break;
            auto ppl = people_random(n, coord_lim);
            long long q_cap = max(1LL, rem / n);
            long long q_goal = min<long long>(q_cap, 60000LL);
            DataSet ds;
            ds.n = n;
            ds.x = ppl;
            int rsum = 0;
            for (long long i = 0; i < q_goal; i++) {
                Query q;
                if (i % 5 == 0) q = make_query_r0(ppl, coord_lim);
                else if (i % 7 == 0)
                    q = make_query_rk(ppl, coord_lim, max(0, (int) (budget.sum_r_limit - budget.used_r - rsum)));
                else if (i % 11 == 0)
                    q = make_query_fixed_exactXi(ppl, rnd.next(1, min(30, n)), coord_lim,
                                                 max(0, (int) (budget.sum_r_limit - budget.used_r - rsum)));
                else q = make_query_general(ppl, coord_lim, max(0, (int) (budget.sum_r_limit - budget.used_r - rsum)));
                if (!budget.canAddOneQuery(n, (int) ds.queries.size(), rsum, q.r)) break;
                rsum += q.r;
                ds.queries.push_back(std::move(q));
            }
            if (ds.queries.empty()) ds.queries.push_back(make_query_r0(ppl, coord_lim));
            ds.q = (int) ds.queries.size();
            budget.addDS(ds.n, ds.q, rsum);
            push(std::move(ds));
        }
            break;

        case STRESS: {
            int blocks = rnd.next(1, 3);
            for (int b = 0; b < blocks; b++) {
                long long rem = budget.remProd();
                if (rem <= 0) break;
                int n_try = rnd.next(20000, 80000);
                int n = min<int>(n_try, (int) rem);
                if (!budget.canStartDS(n)) break;
                auto ppl = people_random(n, coord_lim);
                long long q_cap = max(1LL, rem / n);
                long long q_goal = min<long long>(q_cap, 3000LL);
                DataSet ds;
                ds.n = n;
                ds.x = ppl;
                int rsum = 0;
                for (long long i = 0; i < q_goal; i++) {
                    Query q;
                    int typ = rnd.next(1, 100);
                    if (typ <= 25) q = make_query_r0(ppl, coord_lim);
                    else if (typ <= 70)
                        q = make_query_rk(ppl, coord_lim, max(0, (int) (budget.sum_r_limit - budget.used_r - rsum)));
                    else
                        q = make_query_general(ppl, coord_lim,
                                               max(0, (int) (budget.sum_r_limit - budget.used_r - rsum)));
                    if (!budget.canAddOneQuery(n, (int) ds.queries.size(), rsum, q.r)) break;
                    rsum += q.r;
                    ds.queries.push_back(std::move(q));
                }
                if (ds.queries.empty()) ds.queries.push_back(make_query_r0(ppl, coord_lim));
                ds.q = (int) ds.queries.size();
                budget.addDS(ds.n, ds.q, rsum);
                push(std::move(ds));
                if (budget.remProd() <= 0) break;
            }
        }
            break;

        case RKMIX: {
            long long rem = budget.remProd();
            if (rem <= 0) break;
            int n_try = rnd.next(2000, 20000);
            int n = min<int>(n_try, (int) rem);
            if (!budget.canStartDS(n)) break;
            auto ppl = people_random(n, coord_lim);
            long long q_cap = max(1LL, rem / n);
            long long q_goal = min<long long>(q_cap, 5000LL);
            DataSet ds;
            ds.n = n;
            ds.x = ppl;
            int rsum = 0;
            for (long long i = 0; i < q_goal; i++) {
                Query q = (i % 2 == 0) ? make_query_r0(ppl, coord_lim)
                                       : make_query_rk(ppl, coord_lim,
                                                       max(0, (int) (budget.sum_r_limit - budget.used_r - rsum)));
                if (!budget.canAddOneQuery(n, (int) ds.queries.size(), rsum, q.r)) break;
                rsum += q.r;
                ds.queries.push_back(std::move(q));
            }
            if (ds.queries.empty()) ds.queries.push_back(make_query_r0(ppl, coord_lim));
            ds.q = (int) ds.queries.size();
            budget.addDS(ds.n, ds.q, rsum);
            push(std::move(ds));
        }
            break;
    }
    return out;
}

/* ---------- main ---------- */
int main(int argc, char *argv[]) {
    registerGen(argc, argv, 1);
    string suite_s = get_opt<string>("suite", "smallrand");
    Suite suite = parseSuite(suite_s);

    long long sum_nq = get_opt<long long>("sum_nq", constants::SUM_NQ_LIMIT);
    long long sum_r = get_opt<long long>("sum_r", constants::SUM_R_LIMIT);
    ll coord_lim = get_opt<long long>("coord", constants::MAX_COORD);
    int want_t = get_opt<int>("t", constants::MAX_T);
    int max_n_small = get_opt<int>("max_n_small", 10);

    Budget budget{sum_nq, sum_r, 0, 0};
    auto sets = build_suite(suite, budget, want_t, coord_lim, max_n_small);

    if (sets.empty()) {
        vector<ll> ppl = people_random(1, min<ll>(coord_lim, 100));
        Budget b2{max(1LL, sum_nq), max(0LL, sum_r), 0, 0};
        auto ds = make_dataset_generic(ppl, 1, b2, coord_lim, 0);
        print_all({ds});
        return 0;
    }
    print_all(sets);
    return 0;
}
